//
//  main.c
//  ISO9141-2 Probe
//  Copyright(C)2012 Toyohiko Togashi tog001@nifty.com
//
//
//  This program is free software; you can redistribute it and/or modify it under the terms of the
//  GNU General Public License as published by the Free Software Foundation; either version 3
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
//  without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  See the GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License along with this program.
//  If not, see <http://www.gnu.org/licenses/>
//
//
//	Description
//		ISO 9141-2 protocol scanner. Only 5bps initialyze.
//		Needs the special USB device.
//		That is a FTDI FT232R(USB UART converter) and ISO9141-2 K-Line interface circuit.
//
//	Usage
//		Install FTDI D2XX driver software.
//		Connect a PC to USB adapter.
//		Connect a USB adapter to vhecle's OBD2 socket.
//		Start up this program.
//
//	Environment:
//		Runtime:
//			Mac OS X 10.4 -
//		Development:
//			Mac OS X 10.5.8
//			Xcode 3.1.2
//
//
//    Update history
//    ---------- ----- -------------------------------------------
//    2012.01.02 v0.0  First cording
//    2012.01.12 v0.1  Commit
//    2012.02.29 v0.1a Bug fix
//

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include "ftd2xx.h"

#define P1_MIN		0			// [us]
#define P2_MIN		25000		// [us]
#define P2_MAX		50000		// [us]
#define P3_MIN		55000		// [us]
#define P4_MIN		5000		// [us]
#define SYNC_TRY	100			// max 300ms (300ms/50ms)

FT_STATUS send_5baud(FT_HANDLE fd, char *dat, size_t len){
	FT_STATUS	rc;
	useconds_t	wt = (useconds_t)(1.0/5*1000000);	// 5bps
	
	for(; len > 0; --len) {
		char	c;
		int	i;
		
		c = *dat++;
		
		// Start bit
		if ((rc = FT_SetBreakOn(fd)) != FT_OK) {
			break;
		}
		usleep(wt);
		
		// Data bits
		for(i = 0; i < 8; i++) {
			if (c & 0x01) {
				rc = FT_SetBreakOff(fd);
			} else {
				rc = FT_SetBreakOn(fd);
			}
			if (rc != FT_OK) {
				break;
			}
			c = c >> 1;
			usleep(wt);
		}
		if (rc != FT_OK) {
			break;
		}
		
		// Stop bit
		if ((rc = FT_SetBreakOff(fd)) != FT_OK) {
			break;
		}
		usleep(wt);
	}
	
	return(rc);
}

FT_STATUS iso9141_init(FT_HANDLE fd, unsigned char keyword[2]){
	FT_STATUS	rc;
	unsigned char	addr	= 0x33;
	
	do {
		DWORD	l;
		int	i;
		unsigned char	c;
		
		// 5 baud method
		usleep(2000);
		rc = send_5baud(fd, &addr, sizeof(addr));
		
		// Recieve sync
		for(i = 0; i < SYNC_TRY; i++) {
			if ((rc = FT_Read(fd, &c, 1, &l)) != FT_OK) {
				break;
			}
			if ((l > 0) && (c == 0x55)) {
				break;
			}
		}
		if (rc != FT_OK) {
			printf("iso9141_init: sync recv err rc=%d\n", rc);
			break;
		}
		if (i >= SYNC_TRY) {
			rc = FT_OTHER_ERROR;		// Time out
			printf("iso9141_init: sync recv time out rc=%d\n", rc);
			break;
		}
		
		// Recieve key word bytes
		if ((rc = FT_Read(fd, keyword, sizeof(keyword), &l)) != FT_OK) {
			printf("iso9141_init: keyword recv err rc=%d\n", rc);
			break;
		}
		if (l != 2) {
			printf("iso9141_init: keyword recv err l=%d\n", l);
			rc = FT_OTHER_ERROR;		// Time out
			break;
		}
		
		// Send key word 2
		usleep(25000);
		c = ~keyword[1];
		if ((rc = FT_Write(fd, &c, 1, &l)) != FT_OK) {
			printf("iso9141_init: ~kw2 send err rc=%d\n", rc);
			break;
		}
		if ((rc = FT_Read(fd, &c, 1, &l)) != FT_OK) {	// Dummy recv
			printf("iso9141_init: ~kw2 dummy recv err rc=%d\n", rc);
			break;
		}
		
		// Recieve my address
		if ((rc = FT_Read(fd, &c, 1, &l)) != FT_OK) {
			printf("iso9141_init: ~addrrecv err rc=%d\n", rc);
			break;
		}
		if (c != (unsigned char)(~addr)) {
			printf("iso9141_init: ERR: ~addr=%02x l=%d\n", c, l);
			rc = FT_OTHER_ERROR;		// Protocol error
			break;
		}
		
		break;
		
	} while(1);
	
	return(rc);
};

FT_STATUS iso9141_send_msg(FT_HANDLE fd, unsigned char *data, size_t len){
	FT_STATUS	rc;
	unsigned char	head[3] = {0x68, 0x6a, 0xf1};
	unsigned char	csum[1];
	int	s;
	int	i;
	
	
	// Calc check sum
	s = 0;
	for(i = 0; i < sizeof(head); i++) {
		s += head[i];
	}
	for(i = 0; i < len; i++) {
		s += data[i];
	}
	csum[0] = s;
	
	// Debug
#ifdef DEBUG
	printf("iso9141_send: dump=");
	for(i = 0; i < sizeof(head); i++) {
		printf("%02x", head[i]);
	}
	printf(" ");
	for(i = 0; i < len; i++) {
		printf("%02x", data[i]);
	}
	printf(" %02x\n", csum[0]);
#endif
	
	// Send 3 part timming
	usleep(P3_MIN);
	do {
		DWORD	l;
		unsigned char	dummy[8];
		
		if ((rc = FT_Write(fd, head, sizeof(head), &l)) != FT_OK) {
			printf("iso9141_send: haed send err rc=%d\n", rc);
			break;
		}
		if ((rc = FT_Read(fd, dummy, sizeof(head), &l)) != FT_OK) {
			printf("iso9141_send: haed dmyrcv err rc=%d\n", rc);
			break;
		}
		usleep(P4_MIN);
		if ((rc = FT_Write(fd, data, len, &l)) != FT_OK) {
			printf("iso9141_send: data send err rc=%d\n", rc);
			break;
		}
		if ((rc = FT_Read(fd, dummy, len, &l)) != FT_OK) {
			printf("iso9141_send: data dmyrcv err rc=%d\n", rc);
			break;
		}
		usleep(P4_MIN);
		if ((rc = FT_Write(fd, csum, sizeof(csum), &l)) != FT_OK) {
			printf("iso9141_send: csum send err rc=%d\n", rc);
			break;
		}
		if ((rc = FT_Read(fd, dummy, sizeof(csum), &l)) != FT_OK) {
			printf("iso9141_send: csum dmyrcv err rc=%d\n", rc);
			break;
		}
		break;
	} while(1);
	
	return(rc);
}

FT_STATUS iso9141_recv_response(FT_HANDLE fd, unsigned char *data, size_t *len){
	FT_STATUS	rc;
	
	*len = 0;
	do {
		DWORD	l;
		unsigned char	buf[11];
		int	s;
		int	i;
		
		// Recv
		if ((rc = FT_Read(fd, buf, sizeof(buf), &l)) != FT_OK) {
			printf("iso9141_recv: read err rc=%d\n", rc);
			break;
		}
		if (l <= 1) {
			rc = FT_OTHER_ERROR;	// Time out
			printf("iso9141_recv: read time out rc=%d len=%d\n", rc, l);
			break;
		}
		
		// Debug
#ifdef DEBUG
		printf("iso9141_recv: dump=");
		for(i = 0; i < (l-1); i++) {
			if (i == 3) {
				printf(" ");
			}
			printf("%02x", buf[i]);
		}
		printf(" %02x\n", buf[l-1]);
#endif
		
		// Sum check
		if (l > 1) {
			s = 0;
			--l;
			for(i = 0; i < l; i++) {
				s += buf[i];
			}
			if ((unsigned char)s != buf[l]) {
				rc = FT_OTHER_ERROR;	// sum error
				break;
			}
		}
		
		// Set response data
		for(i = 3; i < l; i++) {
			*data++ = buf[i];
			(*len)++;
		}
		break;
		
	} while(1);
	
	return(rc);
}

#define	LOC_A	1
#define	LOC_B	2
#define	LOC_C	3
#define	LOC_D	4
#define	LOC_AB	5
#define	LOC_CD	6
void edit_print(int pid, int a, int b, int c, int d) {
	const struct {
		int	pid;
		int 	loc;
		int 	bit;
		char	*format;
		double	offset;
		double	scale;
	} pid_table[] = {
		{0x01, LOC_A,  0x00, "Monitor status since DTCs cleared: %d\n",                           0.0, 1.0},
		{0x01, LOC_A,  0x80, "    Malfunction Indicator Lamp (MIL) Status: ON\n",                 0.0, 1.0},
		{0x01, LOC_B,  0x00, "Status of continuous monitoring tests since DTC cleared: 0x%02x\n", 0.0, 1.0},
		{0x01, LOC_B,  0x01, "    Misfire monitoring: SUPPORTED\n",                               0.0, 1.0},
		{0x01, LOC_B,  0x02, "    Fuel system monitoring: SUPPORTED\n",                           0.0, 1.0},
		{0x01, LOC_B,  0x04, "    Comprehensive component monitoring: SUPPORTED\n",               0.0, 1.0},
		{0x01, LOC_B,  0x10, "    Misfire monitoring: NOT READY\n",                               0.0, 1.0},
		{0x01, LOC_B,  0x20, "    Fuel system monitoring: NOT READY\n",                           0.0, 1.0},
		{0x01, LOC_B,  0x40, "    Comprehensive component monitoring: NOT READY\n",               0.0, 1.0},
		{0x01, LOC_C,  0x00, "Supported tests run at least once per trip: 0x%02x\n",              0.0, 1.0},
		{0x01, LOC_C,  0x01, "    Catalyst monitoring: SUPPORTED\n",                              0.0, 1.0},
		{0x01, LOC_C,  0x02, "    Heated catalyst monitoring: SUPPORTED\n",                       0.0, 1.0},
		{0x01, LOC_C,  0x04, "    Evaporative system monitoring: SUPPORTED\n",                    0.0, 1.0},
		{0x01, LOC_C,  0x08, "    Secondary air system monitoring: SUPPORTED\n",                  0.0, 1.0},
		{0x01, LOC_C,  0x10, "    A/C system refrigerant monitoring: SUPPORTED\n",                0.0, 1.0},
		{0x01, LOC_C,  0x20, "    Oxygen sensor monitoring: SUPPORTED\n",                         0.0, 1.0},
		{0x01, LOC_C,  0x40, "    Oxygen sensor heater monitoring: SUPPORTED\n",                  0.0, 1.0},
		{0x01, LOC_C,  0x80, "    EGR system monitoring: SUPPORTED\n",                            0.0, 1.0},
		{0x01, LOC_D,  0x00, "Status of tests run at least once per trip: 0x%02x\n",              0.0, 1.0},
		{0x01, LOC_D,  0x01, "    Catalyst monitoring: NOT READY\n",                              0.0, 1.0},
		{0x01, LOC_D,  0x02, "    Heated catalyst monitoring: NOT READY\n",                       0.0, 1.0},
		{0x01, LOC_D,  0x04, "    Evaporative system monitoring: NOT READY\n",                    0.0, 1.0},
		{0x01, LOC_D,  0x08, "    Secondary air system monitoring: NOT READY\n",                  0.0, 1.0},
		{0x01, LOC_D,  0x10, "    A/C system refrigerant monitoring: NOT READY\n",                0.0, 1.0},
		{0x01, LOC_D,  0x20, "    Oxygen sensor monitoring: NOT READY\n",                         0.0, 1.0},
		{0x01, LOC_D,  0x40, "    Oxygen sensor heater monitoring: NOT READY\n",                  0.0, 1.0},
		{0x01, LOC_D,  0x80, "    EGR system monitoring: NOT READY\n",                            0.0, 1.0},
		{0x02, LOC_AB, 0x00, "DTC that caused required freeze frame data storage: 0x%04x\n",      0.0, 1.0},
		{0x03, LOC_A,  0x00, "Fuel system 1 status: 0x%02x\n",                                    0.0, 1.0},
		{0x03, LOC_A,  0x01, "     Open loop - has not yet satisfied conditions to go closed loop\n", 0.0, 1.0},
		{0x03, LOC_A,  0x02, "     Closed loop - using oxygen sensor(s) as feedback for fuel control\n", 0.0, 1.0},
		{0x03, LOC_A,  0x04, "     Open loop due to driving conditions\n",                        0.0, 1.0},
		{0x03, LOC_A,  0x08, "     Open loop - due to detected system fault\n",                   0.0, 1.0},
		{0x03, LOC_A,  0x10, "     Closed loop, but fault with at least one oxygen sensor - may be using single oxygen sensor for fuel control\n", 0.0, 1.0},
		{0x03, LOC_B,  0x00, "Fuel system 2 status: 0x%02x\n",                                    0.0, 1.0},
		{0x03, LOC_B,  0x01, "     Open loop - has not yet satisfied conditions to go closed loop\n", 0.0, 1.0},
		{0x03, LOC_B,  0x02, "     Closed loop - using oxygen sensor(s) as feedback for fuel control\n", 0.0, 1.0},
		{0x03, LOC_B,  0x04, "     Open loop due to driving conditions\n",                        0.0, 1.0},
		{0x03, LOC_B,  0x08, "     Open loop - due to detected system fault\n",                   0.0, 1.0},
		{0x03, LOC_B,  0x10, "     Closed loop, but fault with at least one oxygen sensor - may be using single oxygen sensor for fuel control\n", 0.0, 1.0},
		{0x04, LOC_A,  0x00, "Calculated LOAD Value: %d[%%]\n",                                   0.0, 1.0},
		{0x05, LOC_A,  0x00, "Engine Coolant Temperature: %d[^C]\n",                            -40.0, 1.0},
		{0x06, LOC_A,  0x00, "Short Term Fuel Trim - Bank 1: %d[%%]\n",                        -100.0, 0.7813},
		{0x06, LOC_B,  0x00, "Short Term Fuel Trim - Bank 3: %d[%%]\n",                        -100.0, 0.7813},
		{0x07, LOC_A,  0x00, "Long Term Fuel Trim - Bank 1: %d[%%]\n",                         -100.0, 0.7813},
		{0x07, LOC_B,  0x00, "Long Term Fuel Trim - Bank 3: %d[%%]\n",                         -100.0, 0.7813},
		{0x08, LOC_A,  0x00, "Short Term Fuel Trim - Bank 2: %d[%%]\n",                        -100.0, 0.7813},
		{0x08, LOC_B,  0x00, "Short Term Fuel Trim - Bank 4: %d[%%]\n",                        -100.0, 0.7813},
		{0x09, LOC_A,  0x00, "Long Term Fuel Trim - Bank 2: %d[%%]\n",                         -100.0, 0.7813},
		{0x09, LOC_B,  0x00, "Long Term Fuel Trim - Bank 4: %d[%%]\n",                         -100.0, 0.7813},
		{0x0a, LOC_A,  0x00, "Fuel Rail Pressure: %d[kPa]\n",                                     0.0, 3.0},
		{0x0b, LOC_A,  0x00, "Intake Manifold Absolute Pressure: %d[kPa]\n",                      0.0, 3.0},
		{0x0c, LOC_AB, 0x00, "Engine RPM: %d[rpm]\n",                                             0.0, 0.25},
		{0x0d, LOC_A,  0x00, "Vehicle Speed Sensor: %d[km/h]\n",                                  0.0, 1.0},
		{0x0e, LOC_A,  0x00, "Ignition Timing Advance for #1 Cylinder: %d[dig]\n",              -64.0, 0.5},
		{0x0f, LOC_A,  0x00, "Intake Air Temperature: %d[^C]\n",                                -40.0, 1.0},
		{0x10, LOC_AB, 0x00, "Air Flow Rate from Mass Air Flow Sensor: %d[g/s]\n",                0.0, 0.01},
		{0x11, LOC_A,  0x00, "Absolute Throttle Position: %d[%%]\n",                              0.0, 0.3922},
		{0x12, LOC_A,  0x00, "Commanded Secondary Air Status: 0x%02x\n",                          0.0, 1.0},
		{0x12, LOC_A,  0x01, "    upstream of first catalytic converter\n",                       0.0, 1.0},
		{0x12, LOC_A,  0x02, "    downstream of first catalytic converter inlet\n",               0.0, 1.0},
		{0x12, LOC_A,  0x04, "    atmosphere / off\n",                                            0.0, 1.0},
		{0x13, LOC_A,  0x00, "Location of Oxygen Sensors: 0x%02x\n",                              0.0, 1.0},
		{0x13, LOC_A,  0x01, "    Bank 1 - Sensor 1\n",                                           0.0, 1.0},
		{0x13, LOC_A,  0x02, "    Bank 1 - Sensor 2\n",                                           0.0, 1.0},
		{0x13, LOC_A,  0x04, "    Bank 1 - Sensor 3\n",                                           0.0, 1.0},
		{0x13, LOC_A,  0x08, "    Bank 1 - Sensor 4\n",                                           0.0, 1.0},
		{0x13, LOC_A,  0x10, "    Bank 2 - Sensor 1\n",                                           0.0, 1.0},
		{0x13, LOC_A,  0x20, "    Bank 2 - Sensor 2\n",                                           0.0, 1.0},
		{0x13, LOC_A,  0x40, "    Bank 2 - Sensor 3\n",                                           0.0, 1.0},
		{0x13, LOC_A,  0x80, "    Bank 2 - Sensor 4\n",                                           0.0, 1.0},
		{0x14, LOC_A,  0x00, "Bank 1 – Sensor 1 Oxygen Sensor Output Voltage: %d[mV]\n",          0.0, 5.000},
		{0x14, LOC_B,  0x00, "Bank 1 – Sensor 1 Short Term Fuel Trim: %d[%%]\n",               -100.0, 0.7813},
		{0x15, LOC_A,  0x00, "Bank 1 – Sensor 2 Oxygen Sensor Output Voltage: %d[mV]\n",          0.0, 5.000},
		{0x15, LOC_B,  0x00, "Bank 1 – Sensor 2 Short Term Fuel Trim: %d[%%]\n",               -100.0, 0.7813},
		{0x16, LOC_A,  0x00, "Bank 1 – Sensor 3 Oxygen Sensor Output Voltage: %d[mV]\n",          0.0, 5.000},
		{0x16, LOC_B,  0x00, "Bank 1 – Sensor 3 Short Term Fuel Trim: %d[%%]\n",               -100.0, 0.7813},
		{0x17, LOC_A,  0x00, "Bank 1 – Sensor 4 Oxygen Sensor Output Voltage: %d[mV]\n",          0.0, 5.000},
		{0x17, LOC_B,  0x00, "Bank 1 – Sensor 4 Short Term Fuel Trim: %d[%%]\n",               -100.0, 0.7813},
		{0x18, LOC_A,  0x00, "Bank 2 – Sensor 1 Oxygen Sensor Output Voltage: %d[mV]\n",          0.0, 5.000},
		{0x18, LOC_B,  0x00, "Bank 2 – Sensor 1 Short Term Fuel Trim: %d[%%]\n",               -100.0, 0.7813},
		{0x19, LOC_A,  0x00, "Bank 2 – Sensor 2 Oxygen Sensor Output Voltage: %d[mV]\n",          0.0, 5.000},
		{0x19, LOC_B,  0x00, "Bank 2 – Sensor 2 Short Term Fuel Trim: %d[%%]\n",               -100.0, 0.7813},
		{0x1a, LOC_A,  0x00, "Bank 2 – Sensor 3 Oxygen Sensor Output Voltage: %d[mV]\n",          0.0, 5.000},
		{0x1a, LOC_B,  0x00, "Bank 2 – Sensor 3 Short Term Fuel Trim: %d[%%]\n",               -100.0, 0.7813},
		{0x1b, LOC_A,  0x00, "Bank 2 – Sensor 4 Oxygen Sensor Output Voltage: %d[mV]\n",          0.0, 5.000},
		{0x1b, LOC_B,  0x00, "Bank 2 – Sensor 4 Short Term Fuel Trim: %d[%%]\n",               -100.0, 0.7813},
		{0x1c, LOC_A,  0x00, "OBD requirements to which vehicle is designed: 0x%02x\n",           0.0, 1.0},
		{0x1d, LOC_A,  0x00, "Location of oxygen sensors: 0x%02x\n",                              0.0, 1.0},
		{0x1d, LOC_A,  0x01, "    Bank 1 - Sensor 1\n",                                           0.0, 1.0},
		{0x1d, LOC_A,  0x02, "    Bank 1 - Sensor 2\n",                                           0.0, 1.0},
		{0x1d, LOC_A,  0x04, "    Bank 2 - Sensor 1\n",                                           0.0, 1.0},
		{0x1d, LOC_A,  0x08, "    Bank 2 - Sensor 2\n",                                           0.0, 1.0},
		{0x1d, LOC_A,  0x10, "    Bank 3 - Sensor 1\n",                                           0.0, 1.0},
		{0x1d, LOC_A,  0x20, "    Bank 3 - Sensor 2\n",                                           0.0, 1.0},
		{0x1d, LOC_A,  0x40, "    Bank 4 - Sensor 1\n",                                           0.0, 1.0},
		{0x1d, LOC_A,  0x80, "    Bank 4 - Sensor 2\n",                                           0.0, 1.0},
		{0x1e, LOC_A,  0x00, "Auxiliary Input Status: 0x%02x\n",                                  0.0, 1.0},
		{0x1e, LOC_A,  0x01, "    Power Take Off (PTO) : ON\n",                                   0.0, 1.0},
		{0x1f, LOC_AB, 0x00, "Time Since Engine Start: %d[sec]\n",                                0.0, 1.0},
		{0x21, LOC_AB, 0x00, "Distance Travelled While MIL is Activated: %d[km]\n",               0.0, 1.0},
		{0x22, LOC_AB, 0x00, "Fuel Rail Pressure relative to manifold vacuum: %d[kPa]\n",         0.0, 0.079},
		{0x23, LOC_AB, 0x00, "Fuel Rail Pressure: %d[kPa]\n",                                     0.0,10.0},
		{0x24, LOC_AB, 0x00, "Bank 1 – Sensor 1 Equivalence Ratio (lambda): %d/1000\n",           0.0, 0.0305},
		{0x24, LOC_CD, 0x00, "Bank 1 – Sensor 1 Oxygen Sensor Voltage: %d[mV]\n",                 0.0, 0.122},
		{0x25, LOC_AB, 0x00, "Bank 1 – Sensor 2 Equivalence Ratio (lambda): %d/1000\n",           0.0, 0.0305},
		{0x25, LOC_CD, 0x00, "Bank 1 – Sensor 2 Oxygen Sensor Voltage: %d[mV]\n",                 0.0, 0.122},
		{0x26, LOC_AB, 0x00, "Bank 1 – Sensor 3 Equivalence Ratio (lambda): %d/1000\n",           0.0, 0.0305},
		{0x26, LOC_CD, 0x00, "Bank 1 – Sensor 3 Oxygen Sensor Voltage: %d[mV]\n",                 0.0, 0.122},
		{0x27, LOC_AB, 0x00, "Bank 1 – Sensor 4 Equivalence Ratio (lambda): %d/1000\n",           0.0, 0.0305},
		{0x27, LOC_CD, 0x00, "Bank 1 – Sensor 4 Oxygen Sensor Voltage: %d[mV]\n",                 0.0, 0.122},
		{0x28, LOC_AB, 0x00, "Bank 2 – Sensor 1 Equivalence Ratio (lambda): %d/1000\n",           0.0, 0.0305},
		{0x28, LOC_CD, 0x00, "Bank 2 – Sensor 1 Oxygen Sensor Voltage: %d[mV]\n",                 0.0, 0.122},
		{0x29, LOC_AB, 0x00, "Bank 2 – Sensor 2 Equivalence Ratio (lambda): %d/1000\n",           0.0, 0.0305},
		{0x29, LOC_CD, 0x00, "Bank 2 – Sensor 2 Oxygen Sensor Voltage: %d[mV]\n",                 0.0, 0.122},
		{0x2a, LOC_AB, 0x00, "Bank 2 – Sensor 3 Equivalence Ratio (lambda): %d/1000\n",           0.0, 0.0305},
		{0x2a, LOC_CD, 0x00, "Bank 2 – Sensor 3 Oxygen Sensor Voltage: %d[mV]\n",                 0.0, 0.122},
		{0x2b, LOC_AB, 0x00, "Bank 2 – Sensor 4 Equivalence Ratio (lambda): %d/1000\n",           0.0, 0.0305},
		{0x2b, LOC_CD, 0x00, "Bank 2 – Sensor 4 Oxygen Sensor Voltage: %d[mV]\n",                 0.0, 0.122},
		{0x2c, LOC_A,  0x00, "Commanded EGR: %d[%%]\n",                                           0.0, 0.3922},
		{0x2d, LOC_A,  0x00, "EGR Error: %d[%%]\n",                                            -100.0, 0.7813},
		{0x2e, LOC_A,  0x00, "Commanded Evaporative Purge: %d[%%]\n",                             0.0, 0.3922},
		{0x2f, LOC_A,  0x00, "Fuel Level Input: %d[%%]\n",                                        0.0, 0.3922},
		{0x30, LOC_A,  0x00, "Number of warm-ups since diagnostic trouble codes cleared: %d\n",   0.0, 1.0},
		{0x31, LOC_AB, 0x00, "Distance since diagnostic trouble codes cleared: %d[km]\n",         0.0, 1.0},
		{0x32, LOC_AB, 0x00, "Evap System Vapor Pressure: %d[Pa]\n",                              0.0, 0.25},
		{0x33, LOC_A,  0x00, "Barometric Pressure: %d[kPa]\n",                                    0.0, 1.0},
		{0x34, LOC_AB, 0x00, "Bank 1 – Sensor 1 Equivalence Ratio (lambda): %d/1000\n",           0.0, 0.0305},
		{0x34, LOC_CD, 0x00, "Bank 1 – Sensor 1 Oxygen Sensor Current: %d[mA]\n",              -128.0, 0.00390625},
		{0x35, LOC_AB, 0x00, "Bank 1 – Sensor 2 Equivalence Ratio (lambda): %d/1000\n",           0.0, 0.0305},
		{0x35, LOC_CD, 0x00, "Bank 1 – Sensor 2 Oxygen Sensor Current: %d[mA]\n",              -128.0, 0.00390625},
		{0x36, LOC_AB, 0x00, "Bank 2 – Sensor 1 Equivalence Ratio (lambda): %d/1000\n",           0.0, 0.0305},
		{0x36, LOC_CD, 0x00, "Bank 2 – Sensor 1 Oxygen Sensor Current: %d[mA]\n",              -128.0, 0.00390625},
		{0x37, LOC_AB, 0x00, "Bank 2 – Sensor 2 Equivalence Ratio (lambda): %d/1000\n",           0.0, 0.0305},
		{0x37, LOC_CD, 0x00, "Bank 2 – Sensor 2 Oxygen Sensor Current: %d[mA]\n",              -128.0, 0.00390625},
		{0x38, LOC_AB, 0x00, "Bank 3 – Sensor 1 Equivalence Ratio (lambda): %d/1000\n",           0.0, 0.0305},
		{0x38, LOC_CD, 0x00, "Bank 3 – Sensor 1 Oxygen Sensor Current: %d[mA]\n",              -128.0, 0.00390625},
		{0x39, LOC_AB, 0x00, "Bank 3 – Sensor 2 Equivalence Ratio (lambda): %d/1000\n",           0.0, 0.0305},
		{0x39, LOC_CD, 0x00, "Bank 3 – Sensor 2 Oxygen Sensor Current: %d[mA]\n",              -128.0, 0.00390625},
		{0x3a, LOC_AB, 0x00, "Bank 4 – Sensor 1 Equivalence Ratio (lambda): %d/1000\n",           0.0, 0.0305},
		{0x3a, LOC_CD, 0x00, "Bank 4 – Sensor 1 Oxygen Sensor Current: %d[mA]\n",              -128.0, 0.00390625},
		{0x3b, LOC_AB, 0x00, "Bank 4 – Sensor 2 Equivalence Ratio (lambda): %d/1000\n",           0.0, 0.0305},
		{0x3b, LOC_CD, 0x00, "Bank 4 – Sensor 2 Oxygen Sensor Current: %d[mA]\n",              -128.0, 0.00390625},
		{0x3c, LOC_AB, 0x00, "Catalyst Temperature Bank 1, Sensor 1: %d[^C]\n",                 -40.0, 0.1},
		{0x3d, LOC_AB, 0x00, "Catalyst Temperature Bank 2, Sensor 1: %d[^C]\n",                 -40.0, 0.1},
		{0x3e, LOC_AB, 0x00, "Catalyst Temperature Bank 1, Sensor 2: %d[^C]\n",                 -40.0, 0.1},
		{0x3f, LOC_AB, 0x00, "Catalyst Temperature Bank 2, Sensor 2: %d[^C]\n",                 -40.0, 0.1},
		{0x41, LOC_B,  0x00, "Enable status of continuous monitors this monitoring cycle: 0x%02x\n", 0.0, 1.0},
		{0x41, LOC_B,  0x01, "    Misfire monitoring: YES\n",                                     0.0, 1.0},
		{0x41, LOC_B,  0x02, "    Fuel system monitoring: YES\n",                                 0.0, 1.0},
		{0x41, LOC_B,  0x04, "    Comprehensive component monitoring: YES\n",                     0.0, 1.0},
		{0x41, LOC_B,  0x00, "Completion status of continuous monitors this monitoring cycle: 0x%02x\n", 0.0, 1.0},
		{0x41, LOC_B,  0x10, "    Misfire monitoring: YES\n",                                     0.0, 1.0},
		{0x41, LOC_B,  0x20, "    Fuel system monitoring: YES\n",                                 0.0, 1.0},
		{0x41, LOC_B,  0x40, "    Comprehensive component monitoring: YES\n",                     0.0, 1.0},
		{0x41, LOC_C,  0x00, "Enable status of non-continuous monitors this monitoring cycle: 0x%02x\n", 0.0, 1.0},
		{0x41, LOC_C,  0x01, "    Catalyst monitoring: YES\n",                                    0.0, 1.0},
		{0x41, LOC_C,  0x02, "    Heated catalyst monitoring: YES\n",                             0.0, 1.0},
		{0x41, LOC_C,  0x04, "    Evaporative system monitoring: YES\n",                          0.0, 1.0},
		{0x41, LOC_C,  0x08, "    Secondary air system monitoring: YES\n",                        0.0, 1.0},
		{0x41, LOC_C,  0x10, "    A/C system refrigerant monitoring: YES\n",                      0.0, 1.0},
		{0x41, LOC_C,  0x20, "    Oxygen sensor monitoring: YES\n",                               0.0, 1.0},
		{0x41, LOC_C,  0x40, "    Oxygen sensor heater monitoring: YES\n",                        0.0, 1.0},
		{0x41, LOC_C,  0x80, "    EGR system monitoring: YES\n",                                  0.0, 1.0},
		{0x41, LOC_D,  0x00, "Completion status of non-continuous monitors this monitoring cycle: 0x%02x\n", 0.0, 1.0},
		{0x41, LOC_D,  0x01, "    Catalyst monitoring: YES\n",                                    0.0, 1.0},
		{0x41, LOC_D,  0x02, "    Heated catalyst monitoring: YES\n",                             0.0, 1.0},
		{0x41, LOC_D,  0x04, "    Evaporative system monitoring: YES\n",                          0.0, 1.0},
		{0x41, LOC_D,  0x08, "    Secondary air system monitoring: YES\n",                        0.0, 1.0},
		{0x41, LOC_D,  0x10, "    A/C system refrigerant monitoring: YES\n",                      0.0, 1.0},
		{0x41, LOC_D,  0x20, "    Oxygen sensor monitoring: YES\n",                               0.0, 1.0},
		{0x41, LOC_D,  0x40, "    Oxygen sensor heater monitoring: YES\n",                        0.0, 1.0},
		{0x41, LOC_D,  0x80, "    EGR system monitoring: YES\n",                                  0.0, 1.0},
		{0x42, LOC_AB, 0x00, "Control module voltage: %d[mV]\n",                                  0.0, 1.0},
		{0x43, LOC_AB, 0x00, "Absolute Load Value: %d[%%]\n",                                     0.0, 0.3922},
		{0x44, LOC_AB, 0x00, "Commanded Equivalence Ratio: %d/1000\n",                            0.0, 0.0305},
		{0x45, LOC_A,  0x00, "Relative Throttle Position: %d[%%]\n",                              0.0, 0.3922},
		{0x46, LOC_A,  0x00, "Ambient air temperature: %d[^C]\n",                               -40.0, 1.0},
		{0x47, LOC_A,  0x00, "Absolute Throttle Position B: %d[%%]\n",                            0.0, 0.3922},
		{0x48, LOC_A,  0x00, "Absolute Throttle Position C: %d[%%]\n",                            0.0, 0.3922},
		{0x49, LOC_A,  0x00, "Accelerator Pedal Position D: %d[%%]\n",                            0.0, 0.3922},
		{0x4a, LOC_A,  0x00, "Accelerator Pedal Position E: %d[%%]\n",                            0.0, 0.3922},
		{0x4b, LOC_A,  0x00, "Accelerator Pedal Position E: %d[%%]\n",                            0.0, 0.3922},
		{0x4c, LOC_A,  0x00, "Commanded Throttle Actuator Control: %d[%%]\n",                     0.0, 0.3922},
		{0x4d, LOC_AB, 0x00, "Minutes run by the engine while MIL activated: %d[min]\n",          0.0, 1.0},
		{0x4e, LOC_AB, 0x00, "Time since diagnostic trouble codes cleared: %d[min]\n",            0.0, 1.0},
		{0x00, 0, 0, NULL, 0.0, 0.0}
	};
	int	i;
	double	x;
	int	v;
	int	f;
	
	f = 0;
	for(i = 0; pid_table[i].pid != 0; i++) {
		if (pid_table[i].pid > pid) {
			break;
		} else if (pid_table[i].pid == pid) {
			switch(pid_table[i].loc) {
				case LOC_A:
					v = a;
					break;
				case LOC_B:
					v = b;
					break;
				case LOC_C:
					v = c;
					break;
				case LOC_D:
					v = d;
					break;
				case LOC_AB:
					v = a << 8 | b;
					break;
				case LOC_CD:
					v = c << 8 | d;
					break;
				default:
					v = 0;
					break;
			}
			
			if ((pid_table[i].bit == 0) || (v & pid_table[i].bit)) {
				x = (double)v * pid_table[i].scale + pid_table[i].offset;
				
				printf("0x%02x 0x%02x%02x%02x%02x: ", pid, a, b, c, d);
				printf(pid_table[i].format, (int)x);
			}
			
			f = 1;
		}
	}
	
	if (!f) {
		printf("0x%02x 0x%02x%02x%02x%02x: ?\n", pid, a, b, c, d);
	}
	return;
}

int main() {
	FT_STATUS	rc;
	FT_HANDLE	fd;
	char 		dev[64];
	int		i;
	size_t		len;
	unsigned char	pid_map[32];	// [0].b7=0x01 [0].b6=0x02 ... [0].b0=0x08 [1].b7=0x09 [1].b6=0x0a ...
	unsigned char	keyword[2];
	unsigned char	buf[16];
	
	//
	// Init
	//
	printf("\n");
	printf("*** ISO 9141-2 online dump tool ***\n");
	printf("   Use FT232R + K-LineInterface\n");
	printf("\n");
	fd = NULL;
	
	//
	// Connect to USB -> FT232R -> OBD2 -> ISO-9141-2
	//
	
	if((rc = FT_ListDevices((PVOID)0, dev, FT_LIST_BY_INDEX | FT_OPEN_BY_SERIAL_NUMBER)) != FT_OK) {
		printf("ERROR: Not found FT232R USB UART device (code=%d)\n", rc);
		exit(1);
	}
	
	printf("USB device '%s' open...\n", dev);
	if((rc = FT_Open(0, &fd)) != FT_OK){
		printf("ERROR: USB: FT_Open('%s')=%d\n", dev, rc);
		exit(1);
	}

//	if ((rc = FT_SetBaudRate(fd, 10500)) != FT_OK) {		// DEL 2012.2.29
	if ((rc = FT_SetBaudRate(fd, 10400)) != FT_OK) {		// ADD 2012.2.29
//		printf("ERROR: USB: FT_SetBaudRate('%s',%d)=%d\n", dev, 10500, rc);		// DEL 2012.2.29
		printf("ERROR: USB: FT_SetBaudRate('%s',%d)=%d\n", dev, 10400, rc);		// ADD 2012.2.29
		exit(1);
	}
	
	if ((rc = FT_SetDataCharacteristics(fd, FT_BITS_8, FT_STOP_BITS_1, FT_PARITY_NONE)) != FT_OK) {
		printf("ERROR: USB: FT_SetDataCharacteristics('%s',8BIT,1STOP,NON_PARITY)=%d\n", dev, rc);
		exit(1);
	}
	
	if ((rc = FT_SetFlowControl(fd, FT_FLOW_NONE, 0, 0)) != FT_OK) {
		printf("ERROR: USB: FT_SetFlowControl('%s',NONE)=%d\n", dev, rc);
		exit(1);
	}
	
	if ((rc = FT_SetTimeouts(fd, P2_MAX/1000, P2_MAX/1000)) != FT_OK) {
		printf("ERROR: USB: FT_SetTimeouts('%s',P2_MAX,P2_MAX)=%d\n", dev, rc);
		exit(1);
	}
	
	// Init
	printf("Connect to vehicle...\n");
	if ((rc = iso9141_init(fd, keyword)) != FT_OK) {
		printf("ERROR: Not response a vehicle (usb='%s',code=%d)\n", dev, rc);
		exit(1);
	}
	printf("Connected!\n", dev);
	if ((keyword[0] == 0x08) && (keyword[1] == 0x08)) {
		printf("Protocol is ISO-9141(P2=25ms)\n");
	} else if ((keyword[0] == 0x94) && (keyword[1] == 0x94)) {
		printf("Protocol is ISO-9141(P2=0ms)\n");
	} else {
		printf("ERROR: Unknown protocol. keyword=0x%02x%02x\n", keyword[0], keyword[1]);
		exit(1);
	}
	
	//
	// Get effective PID list
	//
	for(i = 0; i < sizeof(pid_map); i++) {
		pid_map[i] = 0;
	}
	for(i = 0; i < 0x100; i += 0x20) {
		int	j;
		
		buf[0] = 0x01;		// Request current powertrain diagnostic data request SID
		buf[1] = i;		// PID used to determine PID support for PIDs 01-20
		
		if ((rc = iso9141_send_msg(fd, buf, 2)) != FT_OK) {
			printf("ERROR: Message send (usb='%s',msg0=0x%02x,msg1=0x%02x,code=%d)\n", dev, buf[0], buf[1], rc);
			break;
		}
		if ((rc = iso9141_recv_response(fd, buf, &len)) != FT_OK) {
			printf("ERROR: Response recieve (usb='%s',code=%d,len=%d)\n", dev, rc, (int)len);
			break;
		}
		
		j = i >> 3;
		pid_map[j+0] = buf[2];
		pid_map[j+1] = buf[3];
		pid_map[j+2] = buf[4];
		pid_map[j+3] = buf[5];
		
		if (!(buf[5] & 0x01)) {
			break;
		}
	}
	
#ifdef DEBUG
	pid_map[3] |= 0x01;	// force on pid=0x20
#endif
	
	printf("pid_map=1 2 3 4 5 6 7 8 9 A B C D E F 0");
	for(i = 0; i < sizeof(pid_map); i++) {
		int	j;
		int	k;
		
		if ((i & 0x01) == 0) {
			printf("\n     %02x ", i << 3);
		}
		k = pid_map[i];
		for(j = 0; j < 8; j++) {
			if (k & 0x80) {
				printf("* ");
			} else {
				printf("_ ");
			}
			k = k << 1;
		}
	}
	printf("\n");
	
	//
	// Scan values
	//
	for(i = 0; i < sizeof(pid_map); i++) {
		int	j;
		int	k;
		
		k = pid_map[i];
		for(j = 0; j < 8; j++) {
			if (k & 0x80) {
				int pid;
				
				pid = i << 3 | (j + 1);
				
				buf[0] = 0x01;	// Request current powertrain diagnostic data request SID
				buf[1] = pid;
				
				if ((rc = iso9141_send_msg(fd, buf, 2)) != FT_OK) {
					printf("ERROR: Message send (usb='%s',msg0=0x%02x,msg1=0x%02x,code=%d)\n", dev, buf[0], buf[1], rc);
					exit(1);
				}
				if ((rc = iso9141_recv_response(fd, buf, &len)) != FT_OK) {
					printf("ERROR: Response recieve (usb='%s',code=%d,len=%d)\n", dev, rc, (int)len);
					exit(1);
				}
				
				edit_print(pid, buf[2], buf[3], buf[4], buf[5]);
			}
			k = k << 1;
		}
	}
	
	// Close
	if (fd != NULL) {
		FT_Close(fd);
		fd = NULL;
		printf("USB device '%s' Closed\n", dev);
	}
	
	exit(0);
}
